package com.ejie.ab04b.dao.nora;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;
import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.support.DataAccessUtils;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.ab04b.model.nora.Localidad;
import com.gfi.constants.Constants;
import com.gfi.utils.UtilSQL;

/**
 * LocalidadDaoImpl.
 * 
 * @author GFI
 */
@Repository()
@Transactional()
public class LocalidadDaoImpl implements LocalidadDao {

	private JdbcTemplate jdbcTemplate;

	private static final Logger LOGGER = LoggerFactory
			.getLogger(LocalidadDaoImpl.class);

	private RowMapper<Localidad> rwMap = new RowMapper<Localidad>() {

		public Localidad mapRow(ResultSet resultSet, int rowNum)
				throws SQLException {

			return new Localidad(resultSet.getString("CODPROVINCIA"),
					resultSet.getString("CODMUNICIPIO"),
					resultSet.getString("CODLOCALIDAD"),
					resultSet.getString("DESCRIPCION"));
		}
	};

	/**
	 * Method use to set the datasource.
	 * 
	 *  dataSource
	 *            DataSource
	 *
	 * @param dataSource the new data source
	 */
	@Resource()
	public void setDataSource(DataSource dataSource) {
		this.jdbcTemplate = new JdbcTemplate(dataSource);
	}

	/**
	 * Finds a single row in the Localidad table.
	 * 
	 *  localidad
	 *            Pagination
	 *  Localidad
	 *
	 * @param localidad the localidad
	 * @return the localidad
	 */
	@Transactional(readOnly = true)
	public Localidad find(Localidad localidad) {
		// SELECT - FROM
		StringBuilder query = new StringBuilder(this.getSelectFromQuery(true));

		// WHERE
		query.append(" WHERE T1.COD_LOCALIDAD = ? ");

		StringBuilder traza = new StringBuilder("[LocalidadDaoImpl]: find ");
		traza.append(query.toString());
		traza.append("\n Parametros: ").append(localidad.getCodLocalidad());
		LocalidadDaoImpl.LOGGER.info(traza.toString());

		List<Localidad> localidadList = this.jdbcTemplate.query(
				query.toString(), this.rwMap, localidad.getCodLocalidad());
		return (Localidad) DataAccessUtils.uniqueResult(localidadList);
	}

	/**
	 * Finds a List of rows in the Localidad table.
	 * 
	 *  localidad
	 *            Localidad
	 *  List
	 *
	 * @param localidad the localidad
	 * @return the list
	 */
	@Transactional(readOnly = true)
	public List<Localidad> findAll(Localidad localidad) {
		// SELECT - FROM
		StringBuilder query = new StringBuilder(this.getSelectFromQuery(true));

		// Where clause & Params
		Map<String, ?> mapaWhere = this.getWhereMap(localidad);
		StringBuilder where = new StringBuilder(" WHERE 1=1 ");
		where.append(mapaWhere.get("query"));
		query.append(where);
		query.append(" ORDER BY DESCRIPCION");

		List<?> params = (List<?>) mapaWhere.get("params");

		return (List<Localidad>) this.jdbcTemplate.query(query.toString(),
				this.rwMap, params.toArray());
	}

	/**
	 * Finds a List of rows in the Localidad table.
	 * 
	 *  localidad
	 *            Localidad
	 *  List
	 *
	 * @param localidad the localidad
	 * @return the localidad
	 */
	@Transactional(readOnly = true)
	public Localidad findAllFirst(Localidad localidad) {

		StringBuilder query = new StringBuilder(
				"SELECT T1.COD_PROVINCIA CODPROVINCIA, T1.COD_MUNICIPIO CODMUNICIPIO, T1.COD_LOCALIDAD CODLOCALIDAD, T1.DESCRIPCION DESCRIPCION  FROM LOCALIDAD T1 WHERE "
						+ "T1.COD_PROVINCIA=? and T1.COD_MUNICIPIO=? and UPPER(translate(T1.DESCRIPCION, 'áéíóúÁÉÍÓÚ', 'aeiouAEIOU')) LIKE '%'||UPPER(translate(?, 'áéíóúÁÉÍÓÚ', 'aeiouAEIOU'))||'%' and rownum=1 ");

		query.append(" ORDER BY DESCRIPCION");

		List<Localidad> localidadList = this.jdbcTemplate.query(
				query.toString(), this.rwMap, localidad.getCodProvincia(),
				localidad.getCodMunicipio(), localidad.getDescripcion());

		return (Localidad) DataAccessUtils.uniqueResult(localidadList);
	}

	/**
	 * Counts rows in the Localidad table.
	 * 
	 *  localidad
	 *            Localidad
	 *  Long
	 *
	 * @param localidad the localidad
	 * @return the long
	 */
	@Transactional(readOnly = true)
	public Long findAllCount(Localidad localidad) {
		StringBuilder query = new StringBuilder(
				"SELECT COUNT(1) FROM LOCALIDAD T1 ");

		// Where clause & Params
		Map<String, ?> mapaWhere = this.getWhereMap(localidad);
		StringBuilder where = new StringBuilder(" WHERE 1=1 ");
		where.append(mapaWhere.get("query"));
		query.append(where);

		List<?> params = (List<?>) mapaWhere.get("params");

		return this.jdbcTemplate.queryForLong(query.toString(),
				params.toArray());
	}

	/*
	 * MÉTODOS PRIVADOS
	 */

	/**
	 * Devuelve la Select con todos los campos. Incluye el From si se indica.
	 * 
	 *  includeFrom
	 *            boolean
	 *  String
	 *
	 * @param includeFrom the include from
	 * @return the select from query
	 */
	private String getSelectFromQuery(boolean includeFrom) {
		StringBuilder query = new StringBuilder(
				"SELECT T1.COD_PROVINCIA CODPROVINCIA, T1.COD_MUNICIPIO CODMUNICIPIO, T1.COD_LOCALIDAD CODLOCALIDAD, T1.DESCRIPCION DESCRIPCION ");
		if (includeFrom) {
			query.append(" FROM LOCALIDAD T1 ");
		}

		return query.toString();
	}

	/**
	 * Returns a map with the needed value to create the conditions to filter by
	 * the Localidad entity
	 * 
	 *  localidad
	 *            Localidad Bean with the criteria values to filter by.
	 *  Map created with two keys key query stores the sql query syntax
	 *         key params stores the parameter values to be used in the
	 *         condition sentence.
	 *
	 * @param localidad the localidad
	 * @return the where map
	 */
	private Map<String, ?> getWhereMap(Localidad localidad) {

		StringBuilder where = new StringBuilder(Constants.STRING_BUILDER_INIT);
		List<Object> params = new ArrayList<Object>();

		if (localidad != null) {
			if (localidad.getCodLocalidad() != null) {
				UtilSQL.createFilterSQL(where, params, "T1.COD_LOCALIDAD",
						localidad.getCodLocalidad());
			}
			if (localidad.getCodMunicipio() != null) {
				UtilSQL.createFilterSQL(where, params, "T1.COD_MUNICIPIO",
						localidad.getCodMunicipio());
			}
			if (localidad.getCodProvincia() != null) {
				UtilSQL.createFilterSQL(where, params, "T1.COD_PROVINCIA",
						localidad.getCodProvincia());
			}
			if (localidad.getDescripcion() != null) {
				UtilSQL.createFilterLikeSQL(where, params, "T1.DESCRIPCION",
						localidad.getDescripcion(), false);
			}
		}

		Map<String, Object> mapWhere = new HashMap<String, Object>();
		mapWhere.put("query", where);
		mapWhere.put("params", params);

		return mapWhere;
	}
}